<?php
/**
 * WPCom Tracking for WP Admin Page Views.
 *
 * @package automattic/jetpack-mu-wpcom
 */

namespace Automattic\Jetpack\Jetpack_Mu_Wpcom\Wpcom_Wpadmin_Page_View;

use WPCOMSH_Support_Session_Detect;

/**
 * Bump a Tracks stat for every wp-admin page view not generated by an Automattician.
 * This is possibly interesting on its own but will also be used as part of an
 * initiative to understand how much supply we're offering for live chat demand.
 * See https://nosaraproject.wordpress.com/2015/03/06/live-chat-stats/
 */
function wpcom_nosara_track_admin_page_views() {
	if ( do_not_track_a11ns() ) {
		return;
	}

	$is_simple_site = defined( 'IS_WPCOM' ) && IS_WPCOM;
	$is_atomic_site = ! $is_simple_site;

	global $current_user, $current_blog, $current_screen;

	if (
		! $current_user instanceof \WP_User ||
		! $current_screen instanceof \WP_Screen
	) {
		return;
	}

	$blog_id    = null;
	$user_types = array();

	if ( $is_simple_site ) {
		if ( ! $current_blog instanceof \WP_Site ) {
			return;
		}

		$is_calypsoify_enabled = isset( $_GET['calypsoify'] ) && 1 === (int) $_GET['calypsoify']; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		// Calypsoified WP Admin pages should not count as WP Admin page views, since the user is actually in Calypso.
		if ( $is_calypsoify_enabled ) {
			return;
		}

		$blog_id    = $current_blog->blog_id;
		$user_types = \WPCOM_User::get_types();
	}

	if ( $is_atomic_site ) {
		$blog_id    = _wpcom_get_current_blog_id();
		$user_types = wpcom_atomic_get_user_types();
	}
	?>
	<script type="text/javascript">
		var _admin_pv_props = {
			from_page: '<?php echo esc_js( $current_screen->id ); ?>',
			is_block_editor: '<?php echo esc_js( $current_screen->is_block_editor ? 'true' : 'false' ); ?>',
			source: 'wp-admin',
			blog_id: '<?php echo esc_js( (string) $blog_id ); ?>',
			user_type: '<?php echo esc_js( implode( ',', $user_types ) ); ?>'
		};
		_tkq = window._tkq || [];
		_tkq.push( [ 'identifyUser', <?php echo (int) $current_user->ID; ?>, '<?php echo esc_js( $current_user->user_login ); ?>' ] );
		_tkq.push( [ 'recordEvent', 'wpcom_admin_page_view', _admin_pv_props ] );
	</script>
	<?php
}
add_action( 'admin_footer', __NAMESPACE__ . '\wpcom_nosara_track_admin_page_views' );

/**
 * Track non-calypso Customizer views if the user is linked from the frontend.
 */
function wpcom_maybe_track_customizer() {
	// not when the Customizer is loaded in Calypso
	if ( isset( $_GET['calypso'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		return;
	}

	// links from frontend
	if ( isset( $_GET['url'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		return wpcom_track_customizer_from_frontend();
	}

	// standard wp-admin
	wpcom_nosara_track_admin_page_views();
}

/**
 * Since links to the Customizer from the frontend aren't properly wp-admin (no wp-admin chrome,
 * you return to the frontend after closing), they shouldn't be counted as wp-admin views.
 */
function wpcom_track_customizer_from_frontend() {
	global $current_user;

	// Since we view so many screens to offer support and do feature dev, I think it's best not to count us.
	if ( do_not_track_a11ns() ) {
		return;
	}
	?>
	<script type="text/javascript">
		_tkq = window._tkq || [];
		_tkq.push( [ 'identifyUser', <?php echo (int) $current_user->ID; ?>, '<?php echo esc_js( $current_user->user_login ); ?>' ] );
		_tkq.push( [ 'recordEvent', 'wpcom_customize_loaded_from_frontend' ] );
	</script>
	<?php
}
add_action( 'customize_controls_print_footer_scripts', __NAMESPACE__ . '\wpcom_maybe_track_customizer' );

/**
 * Determine if the current user should not be tracked.
 *
 * @return bool
 */
function do_not_track_a11ns() {
	$is_simple_site = defined( 'IS_WPCOM' ) && IS_WPCOM;

	if ( $is_simple_site ) {
		return is_automattician() || is_network_admin();
	}

	return wpcom_atomic_maybe_is_a11n();
}

/**
 * Check if the user might be an a11n on Atomic sites.
 *
 * @return bool
 */
function wpcom_atomic_maybe_is_a11n() {
	$is_proxy_atomic    = defined( 'AT_PROXIED_REQUEST' ) && AT_PROXIED_REQUEST;
	$is_support_session = WPCOMSH_Support_Session_Detect::is_probably_support_session();

	return $is_proxy_atomic && ! $is_support_session;
}

/**
 * Retrieves the user types for Atomic sites.
 *
 * @return array An array of user types.
 */
function wpcom_atomic_get_user_types() {
	$user_types = array();

	if ( get_account_age_in_days() <= 14 ) {
		$user_types[] = 'New User';
	}

	$subscriptions = wpcomsh_get_wpcom_active_subscriptions();
	if ( count( $subscriptions ) > 0 ) {
		$user_types[] = 'Paid';
	}

	if ( wpcom_site_has_feature( \WPCOM_Features::PRIORITY_SUPPORT ) ) {
		$user_types[] = 'Business';
	}

	return $user_types;
}

/**
 * Calculate the number of days since the current user registered their account.
 *
 * @global WP_User $current_user The current user object.
 *
 * @return float The age of the account in days.
 */
function get_account_age_in_days() {
	global $current_user;

	return ( time() - strtotime( $current_user->user_registered ) ) / DAY_IN_SECONDS;
}
