import clsx from 'clsx';
import { useContext, useRef, useState } from 'preact/hooks';
import { translate } from '../i18n';
import { VerbumSignals } from '../state';
import { hasSubscriptionOptionsVisible } from '../utils';

interface SettingsButtonProps {
	expanded: boolean;
	toggleSubscriptionTray: ( event: MouseEvent ) => void;
}

const getAvatarUrl = ( url: string, cacheBuster: number ) => {
	return `${ url }?v=${ cacheBuster }`;
};

export const SettingsButton = ( { expanded, toggleSubscriptionTray }: SettingsButtonProps ) => {
	const { userInfo } = useContext( VerbumSignals );
	const subscriptionOptionsVisible = hasSubscriptionOptionsVisible();

	const [ isLoading, setIsLoading ] = useState( false );
	const [ cacheBuster, setCacheBuster ] = useState( new Date().getTime() );
	const timerRef = useRef( null );
	const quickEditorRef = useRef( null );

	const handleOnClick = ( event: MouseEvent ) => {
		if ( subscriptionOptionsVisible ) {
			toggleSubscriptionTray( event );
		}
	};

	const openEditor = async ev => {
		ev.preventDefault();

		if ( ! quickEditorRef.current ) {
			const { default: getQuickEditor } = await import(
				/* webpackChunkName: 'quick-editor' */ './quick-editor'
			);

			quickEditorRef.current = getQuickEditor(
				userInfo.value?.email,
				setIsLoading,
				setCacheBuster,
				timerRef.current
			);
		}

		quickEditorRef.current.open();
	};

	return (
		<>
			<div className="user-settings-button">
				<button
					type="button"
					aria-label={ translate( 'Edit Gravatar' ) }
					onClick={ openEditor }
					className={ clsx( 'verbum-form__profile', isLoading && 'loading' ) }
				>
					<img
						src={ getAvatarUrl( userInfo.value?.avatar, cacheBuster ) }
						alt={ userInfo.value?.name }
						className={ userInfo.value?.avatar_classes }
						loading="lazy"
					/>
				</button>

				{ subscriptionOptionsVisible && (
					<>
						<button
							type="button"
							aria-pressed={ expanded }
							aria-expanded={ expanded }
							className={ clsx( {
								open: expanded,
								'no-subscriptions': ! subscriptionOptionsVisible,
							} ) }
							aria-label={ translate( 'Settings' ) }
							onClick={ handleOnClick }
						>
							<svg
								xmlns="http://www.w3.org/2000/svg"
								width="16"
								height="16"
								viewBox="0 0 16 16"
								fill="none"
							>
								<g opacity={ expanded ? '.6' : '.3' } clipPath="url(#clip0_221_124)">
									<path
										stroke="var(--verbum-font-color)"
										strokeWidth="1.33333"
										strokeLinecap="round"
										strokeLinejoin="round"
										d="M12.9334 10C12.8447 10.2011 12.8182 10.4241 12.8574 10.6404C12.8966 10.8567 12.9997 11.0562 13.1534 11.2134L13.1934 11.2534C13.3174 11.3772 13.4157 11.5242 13.4828 11.6861C13.5499 11.848 13.5845 12.0215 13.5845 12.1967C13.5845 12.3719 13.5499 12.5454 13.4828 12.7073C13.4157 12.8691 13.3174 13.0162 13.1934 13.14C13.0696 13.264 12.9225 13.3623 12.7607 13.4294C12.5988 13.4965 12.4253 13.5311 12.2501 13.5311C12.0749 13.5311 11.9014 13.4965 11.7395 13.4294C11.5776 13.3623 11.4306 13.264 11.3067 13.14L11.2667 13.1C11.1096 12.9463 10.9101 12.8432 10.6938 12.804C10.4775 12.7648 10.2545 12.7913 10.0534 12.88C9.85624 12.9645 9.68807 13.1049 9.56962 13.2837C9.45117 13.4626 9.3876 13.6722 9.38675 13.8867V14C9.38675 14.3536 9.24627 14.6928 8.99622 14.9428C8.74617 15.1929 8.40704 15.3334 8.05341 15.3334C7.69979 15.3334 7.36065 15.1929 7.11061 14.9428C6.86056 14.6928 6.72008 14.3536 6.72008 14V13.94C6.71492 13.7194 6.64349 13.5054 6.51509 13.3258C6.38668 13.1463 6.20724 13.0095 6.00008 12.9334C5.799 12.8446 5.57595 12.8181 5.35969 12.8574C5.14343 12.8966 4.94387 12.9997 4.78675 13.1534L4.74675 13.1934C4.62292 13.3173 4.47587 13.4157 4.314 13.4828C4.15214 13.5499 3.97864 13.5844 3.80341 13.5844C3.62819 13.5844 3.45469 13.5499 3.29283 13.4828C3.13096 13.4157 2.98391 13.3173 2.86008 13.1934C2.73611 13.0695 2.63777 12.9225 2.57067 12.7606C2.50357 12.5987 2.46903 12.4252 2.46903 12.25C2.46903 12.0748 2.50357 11.9013 2.57067 11.7394C2.63777 11.5776 2.73611 11.4305 2.86008 11.3067L2.90008 11.2667C3.05377 11.1096 3.15687 10.91 3.19608 10.6937C3.2353 10.4775 3.20882 10.2544 3.12008 10.0534C3.03557 9.85617 2.89525 9.68801 2.71639 9.56956C2.53753 9.45111 2.32794 9.38754 2.11341 9.38669H2.00008C1.64646 9.38669 1.30732 9.24621 1.05727 8.99616C0.807224 8.74611 0.666748 8.40698 0.666748 8.05335C0.666748 7.69973 0.807224 7.36059 1.05727 7.11055C1.30732 6.8605 1.64646 6.72002 2.00008 6.72002H2.06008C2.28074 6.71486 2.49475 6.64343 2.67428 6.51503C2.85381 6.38662 2.99056 6.20718 3.06675 6.00002C3.15549 5.79894 3.18196 5.57589 3.14275 5.35963C3.10354 5.14336 3.00044 4.94381 2.84675 4.78669L2.80675 4.74669C2.68278 4.62286 2.58443 4.4758 2.51734 4.31394C2.45024 4.15208 2.4157 3.97857 2.4157 3.80335C2.4157 3.62813 2.45024 3.45463 2.51734 3.29277C2.58443 3.1309 2.68278 2.98385 2.80675 2.86002C2.93058 2.73605 3.07763 2.63771 3.23949 2.57061C3.40136 2.50351 3.57486 2.46897 3.75008 2.46897C3.9253 2.46897 4.0988 2.50351 4.26067 2.57061C4.42253 2.63771 4.56958 2.73605 4.69341 2.86002L4.73341 2.90002C4.89053 3.05371 5.09009 3.15681 5.30636 3.19602C5.52262 3.23524 5.74567 3.20876 5.94675 3.12002H6.00008C6.19726 3.03551 6.36543 2.89519 6.48388 2.71633C6.60233 2.53747 6.66589 2.32788 6.66675 2.11335V2.00002C6.66675 1.6464 6.80722 1.30726 7.05727 1.05721C7.30732 0.807163 7.64646 0.666687 8.00008 0.666687C8.3537 0.666687 8.69284 0.807163 8.94289 1.05721C9.19294 1.30726 9.33342 1.6464 9.33342 2.00002V2.06002C9.33427 2.27454 9.39784 2.48414 9.51629 2.663C9.63474 2.84186 9.8029 2.98218 10.0001 3.06669C10.2012 3.15543 10.4242 3.1819 10.6405 3.14269C10.8567 3.10348 11.0563 3.00038 11.2134 2.84669L11.2534 2.80669C11.3772 2.68272 11.5243 2.58437 11.6862 2.51727C11.848 2.45018 12.0215 2.41564 12.1967 2.41564C12.372 2.41564 12.5455 2.45018 12.7073 2.51727C12.8692 2.58437 13.0163 2.68272 13.1401 2.80669C13.264 2.93052 13.3624 3.07757 13.4295 3.23943C13.4966 3.4013 13.5311 3.5748 13.5311 3.75002C13.5311 3.92524 13.4966 4.09874 13.4295 4.26061C13.3624 4.42247 13.264 4.56952 13.1401 4.69335L13.1001 4.73335C12.9464 4.89047 12.8433 5.09003 12.8041 5.30629C12.7649 5.52256 12.7913 5.74561 12.8801 5.94669V6.00002C12.9646 6.1972 13.1049 6.36537 13.2838 6.48382C13.4626 6.60227 13.6722 6.66583 13.8867 6.66669H14.0001C14.3537 6.66669 14.6928 6.80716 14.9429 7.05721C15.1929 7.30726 15.3334 7.6464 15.3334 8.00002C15.3334 8.35364 15.1929 8.69278 14.9429 8.94283C14.6928 9.19288 14.3537 9.33335 14.0001 9.33335H13.9401C13.7256 9.33421 13.516 9.39778 13.3371 9.51623C13.1582 9.63468 13.0179 9.80284 12.9334 10Z"
									/>
									<path
										fill="var(--verbum-font-color)"
										fillRule="evenodd"
										clipRule="evenodd"
										transform={ expanded ? 'rotate(180)' : 'rotate(0)' }
										d="M8.40093 9.69949C8.16687 9.90014 7.82146 9.90014 7.5874 9.69949L5.69069 8.07359C5.43387 7.85344 5.40413 7.46679 5.62426 7.20995C5.84442 6.95308 6.23113 6.92333 6.48798 7.14351L7.7501 8.22543C7.89054 8.34582 8.09779 8.34582 8.23822 8.22543L9.50035 7.14351C9.75721 6.92333 10.1439 6.95307 10.3641 7.20995C10.5842 7.46679 10.5545 7.85344 10.2976 8.07359L8.40093 9.69949Z"
									/>
								</g>
								<defs>
									<clipPath id="clip0_221_124">
										<rect width="16" height="16" />
									</clipPath>
								</defs>
							</svg>
						</button>
					</>
				) }
			</div>

			{ expanded && (
				<a
					href="https://gravatar.com/profile/avatars"
					onClick={ openEditor }
					aria-label={ translate( 'Edit Gravatar' ) }
				>
					{ translate( 'Edit Gravatar' ) }
				</a>
			) }
		</>
	);
};
